# Cash Flow Management System - Project Structure

## Directory Structure

```
cash-flow-system/
├── app/
│   ├── Http/
│   │   └── Controllers/
│   │       ├── DashboardController.php    # Dashboard logic
│   │       ├── CompanyController.php      # Company CRUD
│   │       ├── ProjectController.php      # Project CRUD
│   │       ├── ParticularController.php   # Particular CRUD
│   │       └── CashFlowController.php     # Cash Flow CRUD
│   └── Models/
│       ├── Company.php                    # Company model
│       ├── Project.php                    # Project model
│       ├── Particular.php                 # Particular model
│       ├── CashFlow.php                   # Cash Flow model
│       └── CashFlowItem.php               # Cash Flow Item model
│
├── database/
│   ├── migrations/
│   │   ├── 2024_01_01_000001_create_companies_table.php
│   │   ├── 2024_01_01_000002_create_projects_table.php
│   │   ├── 2024_01_01_000003_create_particulars_table.php
│   │   ├── 2024_01_01_000004_create_cash_flows_table.php
│   │   └── 2024_01_01_000005_create_cash_flow_items_table.php
│   └── seeders/
│       └── DatabaseSeeder.php             # Sample data seeder
│
├── resources/
│   └── views/
│       ├── layouts/
│       │   └── app.blade.php              # Main layout (navbar, sidebar)
│       ├── dashboard/
│       │   └── index.blade.php            # Dashboard view
│       ├── companies/
│       │   ├── index.blade.php            # Companies listing
│       │   ├── create.blade.php           # Create company form
│       │   └── edit.blade.php             # Edit company form
│       ├── projects/
│       │   ├── index.blade.php            # Projects listing
│       │   ├── create.blade.php           # Create project form
│       │   └── edit.blade.php             # Edit project form
│       ├── particulars/
│       │   ├── index.blade.php            # Particulars listing
│       │   ├── create.blade.php           # Create particular form
│       │   └── edit.blade.php             # Edit particular form
│       └── cash-flows/
│           ├── index.blade.php            # Cash flow history
│           ├── create.blade.php           # Create cash flow
│           ├── show.blade.php             # View cash flow statement
│           └── edit.blade.php             # Edit cash flow
│
├── routes/
│   └── web.php                            # Application routes
│
├── public/
│   ├── index.php                          # Entry point
│   ├── .htaccess                          # Apache config
│   ├── css/                               # Custom CSS (if any)
│   └── js/                                # Custom JS (if any)
│
├── config/                                # Laravel configuration files
├── bootstrap/                             # Laravel bootstrap files
├── storage/                               # Logs, cache, uploads
├── vendor/                                # Composer dependencies (after install)
│
├── .env.example                           # Environment template
├── .gitignore                             # Git ignore rules
├── artisan                                # Laravel CLI
├── composer.json                          # PHP dependencies
├── README.md                              # Project documentation
├── INSTALLATION_GUIDE.md                  # Installation instructions
└── PROJECT_STRUCTURE.md                   # This file
```

## Key Files Explained

### Controllers

#### DashboardController.php
- **Purpose**: Handles dashboard display
- **Methods**:
  - `index()`: Shows statistics, recent cash flows, quick actions

#### CompanyController.php
- **Purpose**: Manages company operations
- **Methods**:
  - `index()`: List all companies
  - `create()`: Show create form
  - `store()`: Save new company
  - `edit()`: Show edit form
  - `update()`: Update company
  - `destroy()`: Delete company

#### ProjectController.php
- **Purpose**: Manages project operations
- **Methods**: CRUD operations + `getByCompany()` for AJAX

#### ParticularController.php
- **Purpose**: Manages particulars (inflow/outflow items)
- **Methods**: CRUD operations + `getByCompany()` for AJAX

#### CashFlowController.php
- **Purpose**: Manages cash flow statements
- **Methods**:
  - `index()`: Cash flow history
  - `create()`: Create new cash flow
  - `store()`: Save cash flow with items
  - `show()`: Display cash flow statement
  - `edit()`: Edit cash flow
  - `update()`: Update cash flow
  - `destroy()`: Delete cash flow

### Models

#### Company Model
- **Relationships**:
  - `hasMany` Projects
  - `hasMany` Particulars
  - `hasMany` CashFlows

#### Project Model
- **Relationships**:
  - `belongsTo` Company
  - `hasMany` CashFlows

#### Particular Model
- **Relationships**:
  - `belongsTo` Company

#### CashFlow Model
- **Relationships**:
  - `belongsTo` Company
  - `belongsTo` Project
  - `hasMany` CashFlowItems
- **Methods**:
  - `getMonthlyTotals()`: Calculate totals for each month

#### CashFlowItem Model
- **Relationships**:
  - `belongsTo` CashFlow
  - `belongsTo` Particular

### Views

#### layouts/app.blade.php
- Main layout template
- Contains:
  - Sidebar navigation
  - Topbar
  - Bottom navigation (mobile)
  - All CSS/JS includes
  - Responsive design code

#### dashboard/index.blade.php
- Dashboard with statistics
- Recent cash flows
- Quick actions
- System overview

#### Cash Flow Views
- **create.blade.php**: 
  - Dynamic form generation
  - Multi-month selection
  - Budget vs Actual entry
  - Real-time calculations
  
- **show.blade.php**:
  - Professional statement rendering
  - Monthly columns
  - Totals and calculations
  - Budget vs Actual comparison

### Routes (web.php)

```php
GET  /                           → Dashboard
GET  /companies                  → Companies list
GET  /companies/create           → Create form
POST /companies                  → Store company
GET  /companies/{id}/edit        → Edit form
PUT  /companies/{id}             → Update company
DELETE /companies/{id}           → Delete company

GET  /projects                   → Projects list
// ... similar CRUD routes for projects

GET  /particulars                → Particulars list
// ... similar CRUD routes for particulars

GET  /cash-flows                 → Cash flows list
GET  /cash-flows/create          → Create form
POST /cash-flows                 → Store cash flow
GET  /cash-flows/{id}            → View statement
GET  /cash-flows/{id}/edit       → Edit form
PUT  /cash-flows/{id}            → Update cash flow
DELETE /cash-flows/{id}          → Delete cash flow

// AJAX endpoints
GET  /api/projects/company/{id}     → Get projects by company
GET  /api/particulars/company/{id}  → Get particulars by company
```

## Database Schema

### companies
- id (PK)
- name
- description
- timestamps

### projects
- id (PK)
- company_id (FK)
- name
- status (Active/Inactive)
- description
- timestamps

### particulars
- id (PK)
- company_id (FK)
- type (Inflow/Outflow)
- name
- status (Active/Inactive)
- description
- timestamps

### cash_flows
- id (PK)
- company_id (FK)
- project_id (FK)
- year
- created_by
- timestamps

### cash_flow_items
- id (PK)
- cash_flow_id (FK)
- particular_id (FK)
- particular_name
- type (Inflow/Outflow)
- month (1-12)
- budget (decimal)
- actual (decimal)
- timestamps

## Key Features Implementation

### 1. Dynamic Form Generation
**Location**: `resources/views/cash-flows/create.blade.php`
**How it works**:
1. User selects company → loads projects via AJAX
2. User selects months → displays selected months
3. User clicks "Load Particulars" → loads particulars via AJAX
4. System generates dynamic table with Budget/Actual columns for each month
5. Real-time calculations update totals as user types

### 2. Responsive Design
**Location**: `resources/views/layouts/app.blade.php`
**Features**:
- Desktop: Sidebar navigation
- Mobile: Bottom navigation bar
- Collapsible sidebar on mobile
- Touch-optimized buttons
- Responsive tables

### 3. Cash Flow Calculations
**Location**: `app/Models/CashFlow.php` → `getMonthlyTotals()`
**Calculates**:
- Total Inflows (Budget & Actual)
- Total Outflows (Budget & Actual)
- Net Cash Flow (Inflows - Outflows)
- For each month separately

## Customization Guide

### Change Colors
Edit: `resources/views/layouts/app.blade.php`
```css
:root {
    --primary-color: #4f46e5;     /* Main color */
    --secondary-color: #06b6d4;   /* Accent color */
    --success-color: #10b981;     /* Success actions */
    --danger-color: #ef4444;      /* Delete/danger */
}
```

### Add New Particular Types
Edit: All particular-related files, change enum from:
```php
enum('type', ['Inflow', 'Outflow'])
```
To:
```php
enum('type', ['Inflow', 'Outflow', 'Transfer', 'Adjustment'])
```

### Modify Dashboard Stats
Edit: `app/Http/Controllers/DashboardController.php`
Add new statistics in the `index()` method

### Add Export to Excel/PDF
Install packages:
```bash
composer require maatwebsite/excel
composer require barryvdh/laravel-dompdf
```
Then add export methods to `CashFlowController.php`

## Development Workflow

### Adding a New Feature
1. Create/modify migration if database changes needed
2. Update model with relationships
3. Create/update controller methods
4. Add routes in `routes/web.php`
5. Create/update views
6. Test thoroughly

### Debugging
- Check `storage/logs/laravel.log`
- Enable `APP_DEBUG=true` in `.env`
- Use `dd()` or `dump()` for debugging
- Check browser console for JS errors

## Security Notes

- All routes use CSRF protection
- Form validation in controllers
- SQL injection prevention via Eloquent ORM
- XSS protection via Blade templating
- Mass assignment protection in models

## Performance Tips

- Use eager loading: `Company::with('projects')->get()`
- Cache configuration in production
- Index foreign keys
- Optimize queries with `select()`
- Use pagination for large datasets

---

This structure follows Laravel best practices and MVC architecture for maintainability and scalability.
