<?php

namespace App\Http\Controllers;

use App\Models\CashFlow;
use App\Models\CashFlowItem;
use App\Models\Company;
use App\Models\Project;
use App\Models\Particular;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class CashFlowController extends Controller
{
    public function index()
    {
        $cashFlows = CashFlow::with(['company', 'project'])->latest()->get();
        return view('cash-flows.index', compact('cashFlows'));
    }

    public function create()
    {
        $companies = Company::all();
        $currentYear = date('Y');
        $years = range($currentYear - 5, $currentYear + 5);
        
        return view('cash-flows.create', compact('companies', 'years'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'company_id' => 'required|exists:companies,id',
            'project_id' => 'required|exists:projects,id',
            'year' => 'required|integer',
            'created_by' => 'nullable|string',
            'items' => 'required|array',
            'items.*.particular_id' => 'required|exists:particulars,id',
            'items.*.type' => 'required|in:Inflow,Outflow',
            'items.*.month' => 'required|integer|min:1|max:12',
            'items.*.budget' => 'required|numeric|min:0',
            'items.*.actual' => 'nullable|numeric|min:0',
        ]);

        DB::beginTransaction();
        try {
            $cashFlow = CashFlow::create([
                'company_id' => $validated['company_id'],
                'project_id' => $validated['project_id'],
                'year' => $validated['year'],
                'created_by' => $validated['created_by'] ?? 'Admin',
            ]);

            foreach ($validated['items'] as $item) {
                $particular = Particular::find($item['particular_id']);
                
                CashFlowItem::create([
                    'cash_flow_id' => $cashFlow->id,
                    'particular_id' => $item['particular_id'],
                    'particular_name' => $particular->name,
                    'type' => $item['type'],
                    'month' => $item['month'],
                    'budget' => $item['budget'],
                    'actual' => $item['actual'] ?? 0,
                ]);
            }

            DB::commit();

            return redirect()->route('cash-flows.show', $cashFlow)
                ->with('success', 'Cash flow created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Error creating cash flow: ' . $e->getMessage());
        }
    }

    public function show(CashFlow $cashFlow)
    {
        $cashFlow->load(['company', 'project', 'items']);
        
        $months = [
            1 => 'January', 2 => 'February', 3 => 'March', 4 => 'April',
            5 => 'May', 6 => 'June', 7 => 'July', 8 => 'August',
            9 => 'September', 10 => 'October', 11 => 'November', 12 => 'December'
        ];

        $inflowItems = $cashFlow->items()->where('type', 'Inflow')->get()->groupBy('particular_name');
        $outflowItems = $cashFlow->items()->where('type', 'Outflow')->get()->groupBy('particular_name');
        
        $monthlyTotals = $cashFlow->getMonthlyTotals();

        return view('cash-flows.show', compact('cashFlow', 'months', 'inflowItems', 'outflowItems', 'monthlyTotals'));
    }

    public function edit(CashFlow $cashFlow)
    {
        $cashFlow->load(['items']);
        $companies = Company::all();
        $projects = Project::where('company_id', $cashFlow->company_id)->get();
        $particulars = Particular::where('company_id', $cashFlow->company_id)->get();
        $currentYear = date('Y');
        $years = range($currentYear - 5, $currentYear + 5);

        return view('cash-flows.edit', compact('cashFlow', 'companies', 'projects', 'particulars', 'years'));
    }

    public function update(Request $request, CashFlow $cashFlow)
    {
        $validated = $request->validate([
            'company_id' => 'required|exists:companies,id',
            'project_id' => 'required|exists:projects,id',
            'year' => 'required|integer',
            'created_by' => 'nullable|string',
            'items' => 'required|array',
            'items.*.particular_id' => 'required|exists:particulars,id',
            'items.*.type' => 'required|in:Inflow,Outflow',
            'items.*.month' => 'required|integer|min:1|max:12',
            'items.*.budget' => 'required|numeric|min:0',
            'items.*.actual' => 'nullable|numeric|min:0',
        ]);

        DB::beginTransaction();
        try {
            $cashFlow->update([
                'company_id' => $validated['company_id'],
                'project_id' => $validated['project_id'],
                'year' => $validated['year'],
                'created_by' => $validated['created_by'] ?? 'Admin',
            ]);

            $cashFlow->items()->delete();

            foreach ($validated['items'] as $item) {
                $particular = Particular::find($item['particular_id']);
                
                CashFlowItem::create([
                    'cash_flow_id' => $cashFlow->id,
                    'particular_id' => $item['particular_id'],
                    'particular_name' => $particular->name,
                    'type' => $item['type'],
                    'month' => $item['month'],
                    'budget' => $item['budget'],
                    'actual' => $item['actual'] ?? 0,
                ]);
            }

            DB::commit();

            return redirect()->route('cash-flows.show', $cashFlow)
                ->with('success', 'Cash flow updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Error updating cash flow: ' . $e->getMessage());
        }
    }

    public function destroy(CashFlow $cashFlow)
    {
        $cashFlow->delete();

        return redirect()->route('cash-flows.index')
            ->with('success', 'Cash flow deleted successfully.');
    }
}
