@extends('layouts.app')

@section('title', 'Create Cash Flow')
@section('page-title', 'Create Cash Flow Statement')

@section('content')
<div class="row">
    <div class="col-12">
        <div class="table-card">
            <h5 class="mb-4">Cash Flow Information</h5>

            <form action="{{ route('cash-flows.store') }}" method="POST" id="cashFlowForm">
                @csrf

                <div class="row g-3 mb-4">
                    <div class="col-md-3">
                        <label for="company_id" class="form-label">Company <span class="text-danger">*</span></label>
                        <select class="form-select" id="company_id" name="company_id" required>
                            <option value="">Select Company</option>
                            @foreach($companies as $company)
                                <option value="{{ $company->id }}">{{ $company->name }}</option>
                            @endforeach
                        </select>
                    </div>

                    <div class="col-md-3">
                        <label for="project_id" class="form-label">Project <span class="text-danger">*</span></label>
                        <select class="form-select" id="project_id" name="project_id" required disabled>
                            <option value="">Select Project</option>
                        </select>
                    </div>

                    <div class="col-md-3">
                        <label for="year" class="form-label">Year <span class="text-danger">*</span></label>
                        <select class="form-select" id="year" name="year" required>
                            @foreach($years as $y)
                                <option value="{{ $y }}" {{ $y == date('Y') ? 'selected' : '' }}>{{ $y }}</option>
                            @endforeach
                        </select>
                    </div>

                    <div class="col-md-3">
                        <label for="created_by" class="form-label">Created By</label>
                        <input type="text" class="form-control" id="created_by" name="created_by" value="Admin" placeholder="Your name">
                    </div>
                </div>

                <div class="row g-3 mb-4">
                    <div class="col-md-6">
                        <label for="month_select" class="form-label">Select Months <span class="text-danger">*</span></label>
                        <select class="form-select" id="month_select" multiple size="5">
                            <option value="1">January</option>
                            <option value="2">February</option>
                            <option value="3">March</option>
                            <option value="4">April</option>
                            <option value="5">May</option>
                            <option value="6">June</option>
                            <option value="7">July</option>
                            <option value="8">August</option>
                            <option value="9">September</option>
                            <option value="10">October</option>
                            <option value="11">November</option>
                            <option value="12">December</option>
                        </select>
                        <small class="text-muted">Hold Ctrl/Cmd to select multiple months</small>
                    </div>

                    <div class="col-md-6">
                        <label class="form-label">Selected Months</label>
                        <div id="selected_months_display" class="p-3 bg-light rounded" style="min-height: 120px;">
                            <em class="text-muted">No months selected</em>
                        </div>
                    </div>
                </div>

                <button type="button" class="btn btn-info btn-custom mb-4" id="loadParticulars" disabled>
                    <i class="bi bi-arrow-clockwise me-2"></i>Load Particulars & Generate Form
                </button>

                <div id="cashFlowEntryContainer"></div>

                <div class="mt-4 d-flex gap-2" id="submitButtons" style="display: none !important;">
                    <button type="submit" class="btn btn-primary-custom btn-custom">
                        <i class="bi bi-check-circle me-2"></i>Create Cash Flow
                    </button>
                    <a href="{{ route('cash-flows.index') }}" class="btn btn-secondary btn-custom">
                        <i class="bi bi-x-circle me-2"></i>Cancel
                    </a>
                </div>
            </form>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
$(document).ready(function() {
    let selectedMonths = [];
    let particulars = { inflows: [], outflows: [] };

    $('#company_id').on('change', function() {
        const companyId = $(this).val();
        const projectSelect = $('#project_id');
        const loadBtn = $('#loadParticulars');

        projectSelect.prop('disabled', true).html('<option value="">Loading...</option>');
        loadBtn.prop('disabled', true);
        $('#cashFlowEntryContainer').empty();
        $('#submitButtons').hide();

        if (companyId) {
            $.get('/api/projects/company/' + companyId, function(projects) {
                projectSelect.html('<option value="">Select Project</option>');
                projects.forEach(project => {
                    projectSelect.append(`<option value="${project.id}">${project.name}</option>`);
                });
                projectSelect.prop('disabled', false);
            });
        }
    });

    $('#project_id, #month_select').on('change', function() {
        updateLoadButton();
    });

    function updateLoadButton() {
        const hasProject = $('#project_id').val();
        selectedMonths = $('#month_select').val() || [];
        const hasMonths = selectedMonths.length > 0;

        $('#loadParticulars').prop('disabled', !(hasProject && hasMonths));

        if (hasMonths) {
            const monthNames = ['January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December'];
            const display = selectedMonths.map(m => monthNames[m - 1]).join(', ');
            $('#selected_months_display').html(`<strong>Selected:</strong> ${display}`);
        } else {
            $('#selected_months_display').html('<em class="text-muted">No months selected</em>');
        }
    }

    $('#loadParticulars').on('click', function() {
        const companyId = $('#company_id').val();
        
        $.get('/api/particulars/company/' + companyId, function(data) {
            particulars = data;
            generateCashFlowForm();
        });
    });

    function generateCashFlowForm() {
        const container = $('#cashFlowEntryContainer');
        container.empty();

        if (particulars.inflows.length === 0 && particulars.outflows.length === 0) {
            container.html('<div class="alert alert-warning">No particulars found for this company. Please add some particulars first.</div>');
            return;
        }

        const monthNames = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];

        let html = '<div class="table-responsive"><table class="table table-bordered">';
        html += '<thead class="table-light"><tr>';
        html += '<th width="150">Particulars</th>';
        
        selectedMonths.forEach(month => {
            html += `<th colspan="2" class="text-center">${monthNames[month - 1]}</th>`;
        });
        html += '</tr><tr><th></th>';
        selectedMonths.forEach(() => {
            html += '<th class="text-center" style="font-size: 0.85rem;">Budget</th>';
            html += '<th class="text-center" style="font-size: 0.85rem;">Actual</th>';
        });
        html += '</tr></thead><tbody>';

        // Cash Inflows
        if (particulars.inflows.length > 0) {
            html += '<tr class="table-success"><td colspan="' + (1 + selectedMonths.length * 2) + '"><strong>Cash Inflows</strong></td></tr>';
            
            particulars.inflows.forEach(particular => {
                html += `<tr><td>${particular.name}</td>`;
                selectedMonths.forEach(month => {
                    html += `<td><input type="number" class="form-control form-control-sm inflow-budget" data-month="${month}" step="0.01" min="0" value="0" name="items[${particular.id}_${month}][budget]"></td>`;
                    html += `<td><input type="number" class="form-control form-control-sm inflow-actual" data-month="${month}" step="0.01" min="0" value="0" name="items[${particular.id}_${month}][actual]"></td>`;
                    html += `<input type="hidden" name="items[${particular.id}_${month}][particular_id]" value="${particular.id}">`;
                    html += `<input type="hidden" name="items[${particular.id}_${month}][type]" value="Inflow">`;
                    html += `<input type="hidden" name="items[${particular.id}_${month}][month]" value="${month}">`;
                });
                html += '</tr>';
            });

            html += '<tr class="table-info"><td><strong>Total Inflows</strong></td>';
            selectedMonths.forEach(month => {
                html += `<td class="text-end"><strong class="total-inflow-budget-${month}">0</strong></td>`;
                html += `<td class="text-end"><strong class="total-inflow-actual-${month}">0</strong></td>`;
            });
            html += '</tr>';
        }

        // Cash Outflows
        if (particulars.outflows.length > 0) {
            html += '<tr class="table-danger"><td colspan="' + (1 + selectedMonths.length * 2) + '"><strong>Cash Outflows</strong></td></tr>';
            
            particulars.outflows.forEach(particular => {
                html += `<tr><td>${particular.name}</td>`;
                selectedMonths.forEach(month => {
                    html += `<td><input type="number" class="form-control form-control-sm outflow-budget" data-month="${month}" step="0.01" min="0" value="0" name="items[${particular.id}_${month}][budget]"></td>`;
                    html += `<td><input type="number" class="form-control form-control-sm outflow-actual" data-month="${month}" step="0.01" min="0" value="0" name="items[${particular.id}_${month}][actual]"></td>`;
                    html += `<input type="hidden" name="items[${particular.id}_${month}][particular_id]" value="${particular.id}">`;
                    html += `<input type="hidden" name="items[${particular.id}_${month}][type]" value="Outflow">`;
                    html += `<input type="hidden" name="items[${particular.id}_${month}][month]" value="${month}">`;
                });
                html += '</tr>';
            });

            html += '<tr class="table-info"><td><strong>Total Outflows</strong></td>';
            selectedMonths.forEach(month => {
                html += `<td class="text-end"><strong class="total-outflow-budget-${month}">0</strong></td>`;
                html += `<td class="text-end"><strong class="total-outflow-actual-${month}">0</strong></td>`;
            });
            html += '</tr>';
        }

        // Net Cash Flow
        html += '<tr class="table-primary"><td><strong>Net Cash Flow</strong></td>';
        selectedMonths.forEach(month => {
            html += `<td class="text-end"><strong class="net-flow-budget-${month}">0</strong></td>`;
            html += `<td class="text-end"><strong class="net-flow-actual-${month}">0</strong></td>`;
        });
        html += '</tr>';

        html += '</tbody></table></div>';

        container.html(html);
        $('#submitButtons').show();

        attachCalculationEvents();
    }

    function attachCalculationEvents() {
        $('input[type="number"]').on('input', function() {
            calculateTotals();
        });
    }

    function calculateTotals() {
        selectedMonths.forEach(month => {
            let inflowBudget = 0, inflowActual = 0;
            let outflowBudget = 0, outflowActual = 0;

            $(`.inflow-budget[data-month="${month}"]`).each(function() {
                inflowBudget += parseFloat($(this).val()) || 0;
            });

            $(`.inflow-actual[data-month="${month}"]`).each(function() {
                inflowActual += parseFloat($(this).val()) || 0;
            });

            $(`.outflow-budget[data-month="${month}"]`).each(function() {
                outflowBudget += parseFloat($(this).val()) || 0;
            });

            $(`.outflow-actual[data-month="${month}"]`).each(function() {
                outflowActual += parseFloat($(this).val()) || 0;
            });

            $(`.total-inflow-budget-${month}`).text(inflowBudget.toFixed(2));
            $(`.total-inflow-actual-${month}`).text(inflowActual.toFixed(2));
            $(`.total-outflow-budget-${month}`).text(outflowBudget.toFixed(2));
            $(`.total-outflow-actual-${month}`).text(outflowActual.toFixed(2));

            const netBudget = inflowBudget - outflowBudget;
            const netActual = inflowActual - outflowActual;

            $(`.net-flow-budget-${month}`).text(netBudget.toFixed(2)).css('color', netBudget < 0 ? 'red' : 'green');
            $(`.net-flow-actual-${month}`).text(netActual.toFixed(2)).css('color', netActual < 0 ? 'red' : 'green');
        });
    }
});
</script>
@endpush
