#!/bin/bash

# Cash Flow Management System - Quick Setup Script
# This script automates the installation process

echo "================================================"
echo "Cash Flow Management System - Quick Setup"
echo "================================================"
echo ""

# Check PHP version
echo "🔍 Checking PHP version..."
php_version=$(php -v | head -n 1 | cut -d " " -f 2 | cut -d "." -f 1-2)
if (( $(echo "$php_version >= 8.2" | bc -l) )); then
    echo "✅ PHP $php_version detected"
else
    echo "❌ PHP 8.2 or higher required (found: $php_version)"
    exit 1
fi

# Check if composer is installed
echo "🔍 Checking Composer..."
if command -v composer &> /dev/null; then
    echo "✅ Composer found"
else
    echo "❌ Composer not found. Please install Composer first."
    echo "Visit: https://getcomposer.org"
    exit 1
fi

# Install dependencies
echo ""
echo "📦 Installing dependencies..."
composer install --no-interaction

# Copy .env file
echo ""
echo "⚙️  Setting up environment..."
if [ ! -f .env ]; then
    cp .env.example .env
    echo "✅ .env file created"
else
    echo "ℹ️  .env file already exists"
fi

# Generate application key
echo ""
echo "🔑 Generating application key..."
php artisan key:generate --ansi

# Database setup prompt
echo ""
echo "📊 Database Setup"
echo "================================================"
read -p "Database name (default: cash_flow_system): " db_name
db_name=${db_name:-cash_flow_system}

read -p "Database username (default: root): " db_user
db_user=${db_user:-root}

read -sp "Database password: " db_pass
echo ""

# Update .env file
echo ""
echo "📝 Updating database configuration..."
sed -i "s/DB_DATABASE=.*/DB_DATABASE=$db_name/" .env
sed -i "s/DB_USERNAME=.*/DB_USERNAME=$db_user/" .env
sed -i "s/DB_PASSWORD=.*/DB_PASSWORD=$db_pass/" .env

# Create database
echo ""
read -p "Do you want to create the database? (y/n): " create_db
if [ "$create_db" = "y" ]; then
    echo "Creating database..."
    mysql -u$db_user -p$db_pass -e "CREATE DATABASE IF NOT EXISTS $db_name CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;"
    echo "✅ Database created"
fi

# Run migrations
echo ""
read -p "Run migrations? (y/n): " run_migrations
if [ "$run_migrations" = "y" ]; then
    echo "Running migrations..."
    php artisan migrate --force
    echo "✅ Migrations completed"
fi

# Seed database
echo ""
read -p "Seed database with sample data? (y/n): " seed_db
if [ "$seed_db" = "y" ]; then
    echo "Seeding database..."
    php artisan db:seed --force
    echo "✅ Database seeded"
fi

# Set permissions
echo ""
echo "🔒 Setting permissions..."
chmod -R 775 storage bootstrap/cache
echo "✅ Permissions set"

# Done
echo ""
echo "================================================"
echo "✅ Installation Complete!"
echo "================================================"
echo ""
echo "To start the development server, run:"
echo "  php artisan serve"
echo ""
echo "Then visit: http://localhost:8000"
echo ""
echo "Default credentials (if seeded):"
echo "  - No authentication required"
echo "  - Sample data included"
echo ""
echo "Need help? Check:"
echo "  - README.md"
echo "  - INSTALLATION_GUIDE.md"
echo "  - PROJECT_STRUCTURE.md"
echo ""
echo "Happy cash flow tracking! 💰"
